# Instr: (name, *args)

import TuringMachine
from yaml import safe_load

class RAM:
    def __init__(self, n, instructions = [], inputs = []):
        self.wordsize = n
        self.instructions = instructions
        self.nreg = self.__check()
        self.inputs = inputs

    def __check(self):
        n = 0
        for inst in self.instructions:
            name, args = inst[0], inst[1:]
            if name == 'copy':
                if len(args) != 2: raise ValueError(f'{name} requires 2 arguments ({len(args)} given)')
                n = max(n,*args)
            elif name in ['inc', 'dec', 'load', 'store']:
                if len(args) != 1: raise ValueError(f'{name} requires 1 arguments ({len(args)} given)')
                n = max(n, *args)
            elif name == 'jump':
                if len(args) != 2: raise ValueError(f'{name} requires 2 arguments ({len(args)} given)')
                n = max(n, args[0])
            elif name == 'stop':
                if len(args) != 0: raise ValueError(f'{name} requires 0 arguments ({len(args)} given)')
            else: 
                raise ValueError(f'Unknown instruction {name}')
        return n+1

    def __repr__(self):
        return f"RAM machine with word size {self.wordsize} and {len(self.instructions)} instructions using {self.nreg} registers"

    def __str__(self):
        L = [f'word size: {self.wordsize}', f'inputs: {self.inputs}', 'instructions:']
        l = 0
        for inst in self.instructions:
            l += 1
            L.append(f'  - {list(inst)} # {l}')
        return '\n'.join(L)

    def __getitem__(self, i):
        assert i < self.nreg, f"The machine only has {self.nreg} registers"
        if i < len(self.inputs): return self.inputs[i]
        return 0

    def __setitem__(self, i, v):
        assert i < len(self.inputs), f"Only input registers R0, ..., R{len(self.inputs)-1} can be set"
        assert v < 1<<self.wordsize, f"Inputs must be < 2**{self.wordsize} = {1<<self.wordsize}"
        self.inputs[i] = v

    @classmethod
    def from_yaml(cls, s):
        if not isinstance(s, str):
            s = s.read()
        doc = safe_load(s)

        n = doc['word size']
        inputs = doc['inputs']
        instructions = doc['instructions']
        return cls(n,[tuple(i) for i in instructions],inputs)

    def run(self, inputs=None, output=[0], stop=False, verbose=False,count=False):
        if inputs is None: 
            inputs = self.inputs
        else:
            assert len(inputs) == len(self.inputs), f"Incorrect number of inputs ({len(inputs)} instead of {len(self.inputs)})"
        R = inputs + [0] * (self.nreg-len(inputs))
        if verbose:
            print(R)
        pc = 1
        m = 1<<self.wordsize
        if count: c = 0
        while pc <= len(self.instructions):
            if count: c += 1
            inst = self.instructions[pc-1]
            name = inst[0]
            oldpc = pc
            pc += 1
            if name == 'inc':
                i = inst[1]
                R[i] = (R[i] + 1) % m
            elif name == 'dec':
                i = inst[1]
                R[i] = (R[i] - 1) % m
            elif name == 'jump':
                i = inst[1]
                if R[i] == 0:
                    pc = inst[2]
            elif name == 'copy':
                i = inst[1]
                j = inst[2]
                R[i] = R[j]
            elif name == 'store':
                i = inst[1]
                j = 0
                R[i] = R[j]
            elif name == 'load':
                i = 0
                j = inst[1]
                R[i] = R[j]
            elif name == 'stop':
                break
            if verbose:
                print(f'{oldpc}. {name}({",".join(str(x) for x in inst[1:])}) : {R}', end='')
                if not stop: print()
            if stop: 
                cmd = input()
                if cmd == 'stop':
                    break
                if cmd == 'continue':
                    stop = False
        return ([R[i] for i in output], c) if count else [R[i] for i in output]

    def reduced(self):
        z = self.nreg
        newinst = [('store',z)]
        Lines = []
        for inst in self.instructions:
            name = inst[0]
            Lines.append(len(newinst)+1)
            if name in ['inc', 'dec']:
                i = inst[1]
                if i == 0: i = z
                newinst.append(('load',i))
                newinst.append((name, 0, *inst[2:]))
                newinst.append(('store', i))
            if name == 'jump':
                i = inst[1]
                if i == 0: i = z
                newinst.append(('load',i))
                newinst.append((name, 0, inst[2]))
            elif name == 'copy':
                i = inst[1]
                j = inst[2]
                if i == 0: i = z 
                if j == 0: j = z 
                newinst.extend([('load',j),('store',i)])
            elif name == 'stop':
                newinst.append(('load',z))
                newinst.append(inst)
        for i in range(len(newinst)):
            if newinst[i][0] == 'jump':
                line = newinst[i][2]
                newinst[i] = ('jump',0,Lines[line-1])
        return RAM(self.wordsize, newinst, self.inputs) 

    def to_TM(self):
        M = TuringMachine.TuringMachine()
        l = 0
        for inst in self.instructions:
            l += 1
            name = inst[0]
            if l < len(self.instructions):
                new = f'{l+1}{self.instructions[l][0]}'
            if name == 'inc':
                for s in '01': M[f'{l}inc',s] = (s,'R',f'{l}inc')
                M[f'{l}inc','#'] = ('#','L',f'{l}carry')
                M[f'{l}carry','1'] = ('0','L',f'{l}carry')
                M[f'{l}carry','0'] = ('1','L',f'{l}end')
                M[f'{l}carry',' '] = (' ','R',new)
                for s in '01': M[f'{l}end',s] = (s,'L',f'{l}end')
                M[f'{l}end',' '] = (' ','R',new)
            elif name == 'dec':
                for s in '01': M[f'{l}dec',s] = (s,'R',f'{l}dec')
                M[f'{l}dec','#'] = ('#','L',f'{l}carry')
                M[f'{l}carry','1'] = ('0','L',f'{l}end')
                M[f'{l}carry','0'] = ('1','L',f'{l}carry')
                M[f'{l}carry',' '] = (' ','R',new)
                for s in '01': M[f'{l}end',s] = (s,'L',f'{l}end')
                M[f'{l}end',' '] = (' ','R',new)
            elif name == 'jump':
                M[f'{l}jump','0'] = ('0','R',f'{l}jump')
                M[f'{l}jump','1'] = ('1','L',f'{l}False')
                M[f'{l}jump','#'] = ('#','L',f'{l}True')
                for s in '01': M[f'{l}False',s] = (s,'L',f'{l}False')
                M[f'{l}False',' '] = (' ','R',new)
                for s in '01': M[f'{l}True',s] = (s,'L',f'{l}True')
                target = f'{inst[2]}{self.instructions[inst[2]-1][0]}'
                M[f'{l}True',' '] = (' ','R',target)
            elif name == 'load':
                i = inst[1]
                M[f'{l}load','0'] = ('o','R',f'{l}#1')
                M[f'{l}load','1'] = ('i','R',f'{l}#1')
                for j in range(1,i+1):
                    for s in '01': M[f'{l}#{j}',s] = (s,'R',f'{l}#{j}')
                    if j < i: M[f'{l}#{j}','#'] = ('#','R',f'{l}#{j+1}')
                M[f'{l}#{i}','#'] = ('#','R',f'{l}copy')
                M[f'{l}copy','0'] = ('o','L',f'{l}copy0')
                M[f'{l}copy','1'] = ('i','L',f'{l}copy1')
                for s in ' #': M[f'{l}copy',s] = (s,'L',f'{l}end')
                for s in '01#': M[f'{l}copy0',s] = (s,'L',f'{l}copy0')
                for s in 'io': M[f'{l}copy0',s] = ('0','R',f'{l}m')
                for s in '01#': M[f'{l}copy1',s] = (s,'L',f'{l}copy1')
                for s in 'io': M[f'{l}copy1',s] = ('1','R',f'{l}m')
                for s in '01#': M[f'{l}r',s] = (s,'R',f'{l}r')
                M[f'{l}m','0'] = ('o','R',f'{l}r')
                M[f'{l}m','1'] = ('i','R',f'{l}r')
                M[f'{l}m','#'] = ('#','R',f'{l}r')
                M[f'{l}r','o'] = ('0','R',f'{l}copy')
                M[f'{l}r','i'] = ('1','R',f'{l}copy')
                for s in '01#': M[f'{l}end',s] = (s,'L',f'{l}end')
                M[f'{l}end',' '] = (' ','R',new)
            elif name == 'store':
                i = inst[1]
                M[f'{l}store','0'] = ('o','R',f'{l}#1')
                M[f'{l}store','1'] = ('i','R',f'{l}#1')
                for j in range(1,i+1):
                    for s in '01': M[f'{l}#{j}',s] = (s,'R',f'{l}#{j}')
                    if j < i: M[f'{l}#{j}','#'] = ('#','R',f'{l}#{j+1}')
                M[f'{l}#{i}','#'] = ('#','R',f'{l}mark')
                M[f'{l}mark','0'] = ('o','L',f'{l}end')
                M[f'{l}mark','1'] = ('i','L',f'{l}end')
                for s in ' #': M[f'{l}mark', s] = (s, 'L', f'{l}end')
                for s in '01#': M[f'{l}end',s] = (s,'L',f'{l}end')
                M[f'{l}end','i'] = ('1','R',f'{l}copy1m')
                M[f'{l}end','o'] = ('0','R',f'{l}copy0m')
                M[f'{l}end',' '] = (' ','R',new)
                M[f'{l}copy1m','0'] = ('o','R',f'{l}copy1')
                M[f'{l}copy1m','1'] = ('i','R',f'{l}copy1')
                M[f'{l}copy1m','#'] = ('#','R',f'{l}copy1')
                M[f'{l}copy0m','0'] = ('o','R',f'{l}copy0')
                M[f'{l}copy0m','1'] = ('i','R',f'{l}copy0')
                M[f'{l}copy0m','#'] = ('#','R',f'{l}copy0')
                for s in '01#': M[f'{l}copy1',s] = (s,'R',f'{l}copy1')
                for s in 'io': M[f'{l}copy1',s] = ('1','R',f'{l}mark')
                
                for s in '01#': M[f'{l}copy0',s] = (s,'R',f'{l}copy0')
                for s in 'io': M[f'{l}copy0',s] = ('0','R',f'{l}mark')
            elif name == 'stop':
                pass 
            else:
                raise ValueError(f'Only minimal RAM supported (no instruction "{name}")')

        M.init = '1'+self.instructions[0][0]
        n = self.wordsize
        R = [f'{bin(r)[2:]:0>{n}}' for r in self.inputs] + ['0' * n] * (self.nreg-len(self.inputs))
        M.tape = '#'.join(r for r in R)
        return M
